-- Migration: Remove 'disabled' attribute from widget definitions in config.cnft_form
-- Rules:
-- 1. If 'disabled' exists and 'readonly' does not exist -> copy 'disabled' value to 'readonly', then remove 'disabled'
-- 2. If both exist (same or different value) -> keep only 'readonly', remove 'disabled'
-- 3. If only 'readonly' exists -> no change needed

BEGIN;

-- 1) Core function working on jsonb
CREATE OR REPLACE FUNCTION migrate_disabled_to_readonly(input_json jsonb)
  RETURNS jsonb
  LANGUAGE plpgsql
AS $$
DECLARE
  result jsonb;
  key text;
  value jsonb;
  new_widget jsonb;
BEGIN
  IF input_json IS NULL THEN
    RETURN NULL;
  END IF;

  -- ARRAY
  IF jsonb_typeof(input_json) = 'array' THEN
    RETURN (
      SELECT COALESCE(jsonb_agg(migrate_disabled_to_readonly(elem)), '[]'::jsonb)
      FROM jsonb_array_elements(input_json) AS t(elem)
    );
  END IF;

  -- SCALAR
  IF jsonb_typeof(input_json) <> 'object' THEN
    RETURN input_json;
  END IF;

  -- OBJECT
  result := '{}'::jsonb;

  FOR key, value IN SELECT * FROM jsonb_each(input_json) LOOP
      IF key = 'widget' AND jsonb_typeof(value) = 'object' THEN
        new_widget := value;

        IF new_widget ? 'disabled' THEN
          -- copy disabled -> readonly only if readonly missing
          IF NOT (new_widget ? 'readonly') THEN
            new_widget := jsonb_set(new_widget, '{readonly}', new_widget->'disabled', true);
          END IF;

          -- always remove disabled
          new_widget := new_widget - 'disabled';
        END IF;

        result := jsonb_set(result, ARRAY[key], new_widget, true);
      ELSE
        result := jsonb_set(result, ARRAY[key], migrate_disabled_to_readonly(value), true);
      END IF;
    END LOOP;

  RETURN result;
END;
$$;

-- 2) Wrapper for json column (so UPDATE works when schema is type json)
CREATE OR REPLACE FUNCTION migrate_disabled_to_readonly(input_json json)
  RETURNS json
  LANGUAGE sql
AS $$
SELECT migrate_disabled_to_readonly(input_json::jsonb)::json;
$$;

-- Preview changes (uncomment to test)
-- SELECT code,
--        schema as old_schema,
--        migrate_disabled_to_readonly(schema) as new_schema
-- FROM config.cnft_form
-- WHERE code = 'test.proc.tskActTmpl.cod.v1..7738b9';

-- Apply migration to specific row for testing
UPDATE config.cnft_form
SET schema = migrate_disabled_to_readonly(schema);
-- WHERE code = 'test.proc.tskActTmpl.cod.v1..7738b9';

-- After testing, remove the WHERE clause to apply to all rows:
-- UPDATE config.cnft_form
-- SET schema = migrate_disabled_to_readonly(schema);

-- Optional: only touch rows that actually contain "disabled" somewhere
-- UPDATE config.cnft_form
-- SET schema = migrate_disabled_to_readonly(schema)
-- WHERE schema::text LIKE '%"disabled"%';

-- 3) Clean up functions after migration
DROP FUNCTION IF EXISTS migrate_disabled_to_readonly(json);
DROP FUNCTION IF EXISTS migrate_disabled_to_readonly(jsonb);

COMMIT;
