-- Migration: Convert 'privilege' attribute to widget 'hidden' and 'readonly' in config.cnft_form
-- Target column: config.cnft_form.schema (json)
--
-- Rules (per property object that has both "privilege" and "widget"):
-- 1) privilege.read  -> widget.hidden   = "${hasPriv('<read>')}" (set/overwrite)
-- 2) privilege.write -> widget.readonly =
--    - if readonly missing: "${hasPriv('<write>')}"
--    - if readonly is boolean true: keep true
--    - if readonly is boolean false: replace with "${hasPriv('<write>')}"
--    - if readonly is a string expr:
--        "${(<existingExpr>) && hasPriv('<write>')}"
--        where <existingExpr> is unwrapped from ${...} when present
-- 3) Remove privilege attribute from the property object
-- 4) Recurse through whole schema

BEGIN;

-- 1) Core function on jsonb
CREATE OR REPLACE FUNCTION migrate_privilege_to_widget(input_json jsonb)
  RETURNS jsonb
  LANGUAGE plpgsql
AS $$
DECLARE
  result jsonb;
  key text;
  value jsonb;

  obj jsonb;
  widget jsonb;
  priv jsonb;
  priv_read text;
  priv_write text;

  existing_readonly jsonb;
  existing_expr text;
  combined_readonly text;
  hidden_expr text;
BEGIN
  IF input_json IS NULL THEN
    RETURN NULL;
  END IF;

  -- ARRAY
  IF jsonb_typeof(input_json) = 'array' THEN
    RETURN (
      SELECT COALESCE(jsonb_agg(migrate_privilege_to_widget(elem)), '[]'::jsonb)
      FROM jsonb_array_elements(input_json) AS t(elem)
    );
  END IF;

  -- SCALAR
  IF jsonb_typeof(input_json) <> 'object' THEN
    RETURN input_json;
  END IF;

  -- OBJECT: if looks like property (has privilege + widget object), transform it
  IF (input_json ? 'privilege')
    AND (input_json ? 'widget')
    AND jsonb_typeof(input_json->'widget') = 'object'
  THEN
    obj := input_json;
    widget := obj->'widget';
    priv := obj->'privilege';

    priv_read := NULL;
    priv_write := NULL;

    IF jsonb_typeof(priv) = 'object' THEN
      priv_read := priv->>'read';
      priv_write := priv->>'write';
    END IF;

    -- privilege.read -> widget.hidden (overwrite / set)
    IF priv_read IS NOT NULL AND priv_read <> '' THEN
      hidden_expr := '${hasPriv(''' || priv_read || ''')}';
      widget := jsonb_set(widget, '{hidden}', to_jsonb(hidden_expr), true);
      -- Pokud bys chtěl "skrýt když NEMÁ", tak místo toho:
      -- hidden_expr := '${!hasPriv(''' || priv_read || ''')}';
    END IF;

    -- privilege.write -> widget.readonly
    IF priv_write IS NOT NULL AND priv_write <> '' THEN
      existing_readonly := widget->'readonly';

      IF existing_readonly IS NULL THEN
        combined_readonly := '${hasPriv(''' || priv_write || ''')}';
        widget := jsonb_set(widget, '{readonly}', to_jsonb(combined_readonly), true);

      ELSIF jsonb_typeof(existing_readonly) = 'boolean' THEN
        -- keep true, replace false
        IF existing_readonly = 'false'::jsonb THEN
          combined_readonly := '${hasPriv(''' || priv_write || ''')}';
          widget := jsonb_set(widget, '{readonly}', to_jsonb(combined_readonly), true);
        END IF;

      ELSIF jsonb_typeof(existing_readonly) = 'string' THEN
        existing_expr := existing_readonly::text;         -- includes quotes
        existing_expr := trim(both '"' from existing_expr); -- remove quotes

        -- unwrap ${...} when present
        IF left(existing_expr, 2) = '${' AND right(existing_expr, 1) = '}' THEN
          existing_expr := substring(existing_expr from 3 for char_length(existing_expr) - 3);
        END IF;

        combined_readonly := '${(' || existing_expr || ') && hasPriv(''' || priv_write || ''')}';
        widget := jsonb_set(widget, '{readonly}', to_jsonb(combined_readonly), true);

      ELSE
        -- readonly exists but is not boolean/string (object/array/number) -> leave as-is (safer)
        NULL;
      END IF;
    END IF;

    -- write back widget + drop privilege
    obj := jsonb_set(obj, '{widget}', widget, true);
    obj := obj - 'privilege';

    -- recurse into the rest (keep widget as final, don't recurse into it)
    result := '{}'::jsonb;
    FOR key, value IN SELECT * FROM jsonb_each(obj) LOOP
        IF key = 'widget' THEN
          result := jsonb_set(result, ARRAY[key], value, true);
        ELSE
          result := jsonb_set(result, ARRAY[key], migrate_privilege_to_widget(value), true);
        END IF;
      END LOOP;

    RETURN result;
  END IF;

  -- Default: recurse into all fields
  result := '{}'::jsonb;
  FOR key, value IN SELECT * FROM jsonb_each(input_json) LOOP
      result := jsonb_set(result, ARRAY[key], migrate_privilege_to_widget(value), true);
    END LOOP;

  RETURN result;
END;
$$;

-- 2) Wrapper for json column
CREATE OR REPLACE FUNCTION migrate_privilege_to_widget(input_json json)
  RETURNS json
  LANGUAGE sql
AS $$
SELECT migrate_privilege_to_widget(input_json::jsonb)::json;
$$;

-- Preview (doporučuju)
-- SELECT code,
--        schema as old_schema,
--        migrate_privilege_to_widget(schema) as new_schema
-- FROM config.cnft_form
-- WHERE code = 'neco';

-- Apply migration to specific row for testing
UPDATE config.cnft_form
SET schema = migrate_privilege_to_widget(schema);
-- WHERE code = 'test.proc.tskActTmpl.cod.v1..7738b9';

-- Apply to all rows that contain privilege (volitelně)
-- UPDATE config.cnft_form
-- SET schema = migrate_privilege_to_widget(schema)
-- WHERE schema::text LIKE '%"privilege"%';

-- 3) Clean up
DROP FUNCTION IF EXISTS migrate_privilege_to_widget(json);
DROP FUNCTION IF EXISTS migrate_privilege_to_widget(jsonb);

COMMIT;
