-- Migration: Update tsmControls id values in config.cnft_form.schema
-- This migration transforms old control IDs to new namespaced format
-- Example: "severity" -> "Ticket.New.Severity"
-- Uses entity_type column to determine correct mapping for ambiguous fields

BEGIN;

-- Create a temporary mapping table for ID transformations
-- entity_type NULL means the mapping applies to all entity types (unique keys)
CREATE TEMP TABLE id_mapping (
    entity_type TEXT,
    old_id TEXT,
    new_id TEXT NOT NULL,
    PRIMARY KEY (entity_type, old_id)
);

-- ============================================================================
-- Address (entityType: Address)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('Address', 'formattedAddress', 'Address.New.FormattedAddress'),
('Address', 'postOffice', 'Address.New.PostOffice'),
('Address', 'country', 'Address.New.Country'),
('Address', 'region', 'Address.New.Region'),
('Address', 'district', 'Address.New.District'),
('Address', 'city', 'Address.New.City'),
('Address', 'partOfCity', 'Address.New.PartOfCity'),
('Address', 'street', 'Address.New.Street'),
('Address', 'externalId', 'Address.New.ExternalId'),
('Address', 'propertyNumber', 'Address.New.PropertyNumber'),
('Address', 'regId', 'Address.New.RegId'),
('Address', 'verified', 'Address.New.Verified'),
('Address', 'verificationDate', 'Address.New.VerificationDate'),
('Address', 'replacedById', 'Address.New.ReplacedById'),
('Address', 'rejectedConflicts', 'Address.New.RejectedConflicts'),
('Address', 'streetNumberNo', 'Address.New.StreetNumberNo'),
('Address', 'streetNumberLet', 'Address.New.StreetNumberLet'),
('Address', 'lat', 'Address.New.Lat'),
('Address', 'lon', 'Address.New.Lon'),
('Address', 'description', 'Address.New.Description');

-- ============================================================================
-- BillingDocument (entityType: BillingDocument)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('BillingDocument', 'name', 'BillingDocument.New.Name'),
('BillingDocument', 'invoiceDate', 'BillingDocument.New.InvoiceDate'),
('BillingDocument', 'dueDate', 'BillingDocument.New.DueDate'),
('BillingDocument', 'dateOfRealization', 'BillingDocument.New.DateOfRealization'),
('BillingDocument', 'variableSymbol', 'BillingDocument.New.VariableSymbol'),
('BillingDocument', 'customerId', 'BillingDocument.New.CustomerId'),
('BillingDocument', 'accountId', 'BillingDocument.New.AccountId'),
('BillingDocument', 'invoiceNumber', 'BillingDocument.New.InvoiceNumber'),
('BillingDocument', 'billingCurrency', 'BillingDocument.New.BillingCurrency'),
('BillingDocument', 'billingCycle', 'BillingDocument.New.BillingCycle'),
('BillingDocument', 'status', 'BillingDocument.New.Status'),
('BillingDocument', 'description', 'BillingDocument.New.Description');

-- ============================================================================
-- BillingDocumentLine (entityType: BillingDocumentLine)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('BillingDocumentLine', 'name', 'BillingDocumentLine.New.Name'),
('BillingDocumentLine', 'parentDocumentLineId', 'BillingDocumentLine.New.ParentDocumentLineId'),
('BillingDocumentLine', 'billingDocumentId', 'BillingDocumentLine.New.BillingDocumentId'),
('BillingDocumentLine', 'entityInstanceConfigurationId', 'BillingDocumentLine.New.EntityInstanceConfigurationId'),
('BillingDocumentLine', 'vatRate', 'BillingDocumentLine.New.VatRate'),
('BillingDocumentLine', 'priceWithoutVat', 'BillingDocumentLine.New.PriceWithoutVat'),
('BillingDocumentLine', 'sortOrder', 'BillingDocumentLine.New.SortOrder');

-- ============================================================================
-- EntityCatalogCategory (entityType: EntityCatalogCategory)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('EntityCatalogCategory', 'categoryName', 'EntityCatalogCategory.New.CategoryName'),
('EntityCatalogCategory', 'categoryCode', 'EntityCatalogCategory.New.CategoryCode'),
('EntityCatalogCategory', 'entityCatalog', 'EntityCatalogCategory.New.EntityCatalog'),
('EntityCatalogCategory', 'parent', 'EntityCatalogCategory.New.Parent'),
('EntityCatalogCategory', 'entitySpecificationSpecId', 'EntityCatalogCategory.New.EntitySpecificationSpecId'),
('EntityCatalogCategory', 'icon', 'EntityCatalogCategory.New.Icon'),
('EntityCatalogCategory', 'sortOrder', 'EntityCatalogCategory.New.SortOrder'),
('EntityCatalogCategory', 'description', 'EntityCatalogCategory.New.Description');

-- ============================================================================
-- EntityCatalogSpecification (entityType: EntityCatalogSpecification)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('EntityCatalogSpecification', 'specificationName', 'EntityCatalogSpecification.New.SpecificationName'),
('EntityCatalogSpecification', 'specificationCode', 'EntityCatalogSpecification.New.SpecificationCode'),
('EntityCatalogSpecification', 'catalogCategory', 'EntityCatalogSpecification.New.CatalogCategory'),
('EntityCatalogSpecification', 'entityInstanceSpecId', 'EntityCatalogSpecification.New.EntityInstanceSpecId'),
('EntityCatalogSpecification', 'orderLineSpecId', 'EntityCatalogSpecification.New.OrderLineSpecId'),
('EntityCatalogSpecification', 'icon', 'EntityCatalogSpecification.New.Icon'),
('EntityCatalogSpecification', 'sortOrder', 'EntityCatalogSpecification.New.SortOrder'),
('EntityCatalogSpecification', 'description', 'EntityCatalogSpecification.New.Description'),
('EntityCatalogSpecification', 'instantiable', 'EntityCatalogSpecification.New.Instantiable'),
('EntityCatalogSpecification', 'lifecycleStatus', 'EntityCatalogSpecification.New.LifecycleStatus');

-- ============================================================================
-- RegisterValue (entityType: RegisterValue)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('RegisterValue', 'name', 'RegisterValue.New.Name'),
('RegisterValue', 'code', 'RegisterValue.New.Code'),
('RegisterValue', 'order', 'RegisterValue.New.Order'),
('RegisterValue', 'parent', 'RegisterValue.New.Parent'),
('RegisterValue', 'icon', 'RegisterValue.New.Icon'),
('RegisterValue', 'validFrom', 'RegisterValue.New.ValidFrom'),
('RegisterValue', 'validTo', 'RegisterValue.New.ValidTo'),
('RegisterValue', 'description', 'RegisterValue.New.Description');

-- ============================================================================
-- Customer (entityType: Customer)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('Customer', 'customerName', 'Customer.New.Name'),
('Customer', 'legalForm', 'Customer.New.LegalForm'),
('Customer', 'customerSegment', 'Customer.New.Segment'),
('Customer', 'persIdentNum', 'Customer.New.IdentificationNumber'),
('Customer', 'taxIdentNum', 'Customer.New.Vatin'),
('Customer', 'description', 'Customer.New.Description');

-- ============================================================================
-- Comment (entityType: Comment)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('Comment', 'comment', 'Comment.New.Comment');

-- ============================================================================
-- EntityInstanceConfiguration (entityType: EntityInstanceConfiguration)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('EntityInstanceConfiguration', 'name', 'EntityInstanceConfiguration.New.Name'),
('EntityInstanceConfiguration', 'key', 'EntityInstanceConfiguration.New.Key'),
('EntityInstanceConfiguration', 'customerId', 'EntityInstanceConfiguration.New.CustomerId'),
('EntityInstanceConfiguration', 'accountId', 'EntityInstanceConfiguration.New.AccountId'),
('EntityInstanceConfiguration', 'validFrom', 'EntityInstanceConfiguration.New.ValidFrom'),
('EntityInstanceConfiguration', 'validTo', 'EntityInstanceConfiguration.New.ValidTo'),
('EntityInstanceConfiguration', 'description', 'EntityInstanceConfiguration.New.Description'),
('EntityInstanceConfiguration', 'location1', 'EntityInstanceConfiguration.New.Location1'),
('EntityInstanceConfiguration', 'location2', 'EntityInstanceConfiguration.New.Location2');

-- ============================================================================
-- Order (entityType: Order)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('Order', 'key', 'Order.New.Key'),
('Order', 'customerId', 'Order.New.CustomerId'),
('Order', 'accountId', 'Order.New.AccountId'),
('Order', 'entityInstanceConfiguration', 'Order.New.EntityInstanceConfiguration'),
('Order', 'deliveryDateRequested', 'Order.New.DeliveryDateRequested'),
('Order', 'subject', 'Order.New.Subject'),
('Order', 'description', 'Order.New.Description');

-- ============================================================================
-- Stock (entityType: Stock)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('Stock', 'name', 'Stock.New.Name'),
('Stock', 'ownerUserId', 'Stock.New.OwnerUserId'),
('Stock', 'def', 'Stock.New.Def'),
('Stock', 'status', 'Stock.New.Status'),
('Stock', 'description', 'Stock.New.Description');

-- ============================================================================
-- Ticket (entityType: Ticket)
-- ============================================================================
INSERT INTO id_mapping (entity_type, old_id, new_id) VALUES
('Ticket', 'name', 'Ticket.New.Name'),
('Ticket', 'severity', 'Ticket.New.Severity'),
('Ticket', 'priority', 'Ticket.New.Priority'),
('Ticket', 'channel', 'Ticket.New.Channel'),
('Ticket', 'ticketCategory', 'Ticket.New.TicketCategory'),
('Ticket', 'externalId', 'Ticket.New.ExternalId'),
('Ticket', 'creationGroup', 'Ticket.New.CreationGroup'),
('Ticket', 'description', 'Ticket.New.Description');


-- Create a function to recursively update IDs in the layout structure
-- Takes entity_type as parameter for context-aware mapping
CREATE OR REPLACE FUNCTION pg_temp.migrate_layout_ids(layout_item JSONB, p_entity_type TEXT)
RETURNS JSONB AS $
DECLARE
    result JSONB;
    new_id TEXT;
    items_array JSONB;
    updated_items JSONB;
    i INT;
BEGIN
    IF layout_item IS NULL THEN
        RETURN NULL;
    END IF;

    result := layout_item;

    -- If this item has an 'id' field, check if it needs to be migrated
    IF result ? 'id' AND jsonb_typeof(result -> 'id') = 'string' THEN
        -- First try entity-specific mapping
        SELECT m.new_id INTO new_id
        FROM id_mapping m
        WHERE m.entity_type = p_entity_type
          AND m.old_id = result ->> 'id';

        IF new_id IS NOT NULL THEN
            result := jsonb_set(result, '{id}', to_jsonb(new_id));
        END IF;
    END IF;

    -- If this item has nested 'items' array, recursively process them
    IF result ? 'items' AND jsonb_typeof(result -> 'items') = 'array' THEN
        updated_items := '[]'::JSONB;
        FOR i IN 0..jsonb_array_length(result -> 'items') - 1 LOOP
            updated_items := updated_items || jsonb_build_array(
                pg_temp.migrate_layout_ids(result -> 'items' -> i, p_entity_type)
            );
        END LOOP;
        result := jsonb_set(result, '{items}', updated_items);
    END IF;

    RETURN result;
END;
$ LANGUAGE plpgsql;

-- Create a function to process the entire schema
CREATE OR REPLACE FUNCTION pg_temp.migrate_schema(schema_data JSONB, p_entity_type TEXT)
RETURNS JSONB AS $
DECLARE
    result JSONB;
    layout_array JSONB;
    updated_layout JSONB;
    i INT;
BEGIN
    IF schema_data IS NULL THEN
        RETURN NULL;
    END IF;

    result := schema_data;

    -- Process the 'layout' array if it exists
    IF result ? 'layout' AND jsonb_typeof(result -> 'layout') = 'array' THEN
        updated_layout := '[]'::JSONB;
        FOR i IN 0..jsonb_array_length(result -> 'layout') - 1 LOOP
            updated_layout := updated_layout || jsonb_build_array(
                pg_temp.migrate_layout_ids(result -> 'layout' -> i, p_entity_type)
            );
        END LOOP;
        result := jsonb_set(result, '{layout}', updated_layout);
    END IF;

    RETURN result;
END;
$ LANGUAGE plpgsql;

-- ============================================================================
-- Preview changes (uncomment to test before applying)
-- ============================================================================
-- SELECT
--     id,
--     entity_type,
--     schema AS old_schema,
--     pg_temp.migrate_schema(schema::jsonb, entity_type) AS new_schema
-- FROM config.cnft_form
-- WHERE schema IS NOT NULL
--   AND schema::jsonb ? 'layout'
--   AND entity_type IS NOT NULL
--   -- AND id = 'xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx'  -- Uncomment to test on specific form
-- LIMIT 10;

-- ============================================================================
-- Apply the migration
-- ============================================================================
UPDATE config.cnft_form
SET schema = pg_temp.migrate_schema(schema::jsonb, entity_type)::json
WHERE schema IS NOT NULL
  AND schema::jsonb ? 'layout'
  AND entity_type IS NOT NULL
  -- AND id = 'xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx'  -- Uncomment to test on specific form
;

-- Report how many rows were updated
DO $
DECLARE
    updated_count INT;
BEGIN
    GET DIAGNOSTICS updated_count = ROW_COUNT;
    RAISE NOTICE 'Updated % rows in config.cnft_form', updated_count;
END $;

COMMIT;
