-- Migration: Wrap Ticket.Accordions.Description control in its own dtl-fluent-inplace
-- This migration finds the Ticket.Accordions.Description control and wraps it
-- in a separate dtl-fluent-inplace layout, updating accordion content references
-- Handles both cases:
--   1. Description is directly in accordion items (not wrapped)
--   2. Description is inside a dtl-fluent-inplace wrapper (already wrapped with other items)

BEGIN;

-- Recursive function to process the layout structure
CREATE OR REPLACE FUNCTION pg_temp.process_layout_recursive(item JSONB)
RETURNS JSONB AS $func$
DECLARE
    result JSONB;
    items_array JSONB;
    new_items JSONB;
    current_item JSONB;
    processed_item JSONB;
    i INT;
    j INT;
    description_item JSONB;
    description_index INT;
    found_description BOOLEAN;
    found_in_inplace BOOLEAN;
    inner_items JSONB;
    new_inner_items JSONB;
    accordion_config JSONB;
    new_accordions JSONB;
    accordion JSONB;
    content_array JSONB;
    new_content JSONB;
BEGIN
    IF item IS NULL THEN
        RETURN NULL;
    END IF;

    result := item;

    -- Check if this is the dtl-fluent-accordion container we're looking for
    IF result ? 'widget'
       AND result -> 'widget' ->> 'type' = 'dtl-fluent-accordion'
       AND result ? 'items'
       AND jsonb_typeof(result -> 'items') = 'array' THEN

        items_array := result -> 'items';
        new_items := '[]'::JSONB;
        found_description := FALSE;
        found_in_inplace := FALSE;
        description_item := NULL;
        description_index := -1;

        -- First pass: find the Description control
        FOR i IN 0..jsonb_array_length(items_array) - 1 LOOP
            current_item := items_array -> i;

            -- Case 1: Check if Description is directly in accordion items
            IF current_item ->> 'id' = 'Ticket.Accordions.Description' THEN
                description_item := current_item;
                found_description := TRUE;
                found_in_inplace := FALSE;
                description_index := i;
                -- Don't add to new_items yet, we'll wrap it later
                CONTINUE;
            END IF;

            -- Case 2: Check if this is a dtl-fluent-inplace that contains Ticket.Accordions.Description
            IF current_item ? 'widget'
               AND current_item -> 'widget' ->> 'type' = 'dtl-fluent-inplace'
               AND current_item ? 'items'
               AND jsonb_typeof(current_item -> 'items') = 'array' THEN

                inner_items := current_item -> 'items';
                new_inner_items := '[]'::JSONB;

                FOR j IN 0..jsonb_array_length(inner_items) - 1 LOOP
                    IF inner_items -> j ->> 'id' = 'Ticket.Accordions.Description' THEN
                        -- Found the Description control inside inplace, save it for wrapping
                        description_item := inner_items -> j;
                        found_description := TRUE;
                        found_in_inplace := TRUE;
                        description_index := i;
                    ELSE
                        -- Keep other items in the original wrapper
                        new_inner_items := new_inner_items || jsonb_build_array(inner_items -> j);
                    END IF;
                END LOOP;

                -- Add the modified wrapper (now without Description)
                current_item := jsonb_set(current_item, '{items}', new_inner_items);
            END IF;

            new_items := new_items || jsonb_build_array(current_item);
        END LOOP;

        -- If we found the Description, add a new wrapper for it
        IF found_description AND description_item IS NOT NULL THEN
            -- Create new dtl-fluent-inplace wrapper for Description
            new_items := new_items || jsonb_build_array(
                jsonb_build_object(
                    'type', 'layout',
                    'widget', jsonb_build_object('type', 'dtl-fluent-inplace'),
                    'items', jsonb_build_array(description_item)
                )
            );

            result := jsonb_set(result, '{items}', new_items);

            -- Update accordion content references
            -- The new Description wrapper is now at the end of items array
            IF result ? 'config' AND result -> 'config' ? 'accordions' THEN
                accordion_config := result -> 'config' -> 'accordions';
                new_accordions := '[]'::JSONB;

                FOR i IN 0..jsonb_array_length(accordion_config) - 1 LOOP
                    accordion := accordion_config -> i;

                    -- Check if this accordion references the old Description index
                    IF accordion ? 'content' AND jsonb_typeof(accordion -> 'content') = 'array' THEN
                        content_array := accordion -> 'content';
                        new_content := '[]'::JSONB;

                        FOR j IN 0..jsonb_array_length(content_array) - 1 LOOP
                            -- If this references the old index that had Description
                            IF (content_array -> j)::INT = description_index THEN
                                IF found_in_inplace THEN
                                    -- Was in inplace: add both the new wrapper index and the old (now empty) wrapper
                                    new_content := jsonb_build_array(jsonb_array_length(new_items) - 1) || new_content;
                                    new_content := new_content || jsonb_build_array(content_array -> j);
                                ELSE
                                    -- Was direct: just replace with new wrapper index
                                    new_content := new_content || jsonb_build_array(jsonb_array_length(new_items) - 1);
                                END IF;
                            ELSE
                                new_content := new_content || jsonb_build_array(content_array -> j);
                            END IF;
                        END LOOP;

                        accordion := jsonb_set(accordion, '{content}', new_content);
                    END IF;

                    new_accordions := new_accordions || jsonb_build_array(accordion);
                END LOOP;

                result := jsonb_set(result, '{config,accordions}', new_accordions);
            END IF;

            RETURN result;
        ELSE
            -- No changes needed at this level, but still process children
            result := jsonb_set(result, '{items}', new_items);
        END IF;

        RETURN result;
    END IF;

    -- Recursively process 'items' array if present
    IF result ? 'items' AND jsonb_typeof(result -> 'items') = 'array' THEN
        new_items := '[]'::JSONB;
        FOR i IN 0..jsonb_array_length(result -> 'items') - 1 LOOP
            processed_item := pg_temp.process_layout_recursive(result -> 'items' -> i);
            new_items := new_items || jsonb_build_array(processed_item);
        END LOOP;
        result := jsonb_set(result, '{items}', new_items);
    END IF;

    -- Recursively process 'layout' array if present
    IF result ? 'layout' AND jsonb_typeof(result -> 'layout') = 'array' THEN
        new_items := '[]'::JSONB;
        FOR i IN 0..jsonb_array_length(result -> 'layout') - 1 LOOP
            processed_item := pg_temp.process_layout_recursive(result -> 'layout' -> i);
            new_items := new_items || jsonb_build_array(processed_item);
        END LOOP;
        result := jsonb_set(result, '{layout}', new_items);
    END IF;

    RETURN result;
END;
$func$ LANGUAGE plpgsql;

-- ============================================================================
-- Preview changes (uncomment to test before applying)
-- ============================================================================
-- SELECT
--     id,
--     entity_type,
--     schema AS old_schema,
--     pg_temp.process_layout_recursive(schema::jsonb) AS new_schema
-- FROM config.cnft_form
-- WHERE schema IS NOT NULL
--   AND schema::text LIKE '%Ticket.Accordions.Description%'
-- LIMIT 5;

-- ============================================================================
-- Apply the migration
-- ============================================================================
UPDATE config.cnft_form
SET schema = pg_temp.process_layout_recursive(schema::jsonb)::json
WHERE schema IS NOT NULL
  AND schema::text LIKE '%Ticket.Accordions.Description%';

-- Report how many rows were updated
DO $do$
DECLARE
    updated_count INT;
BEGIN
    GET DIAGNOSTICS updated_count = ROW_COUNT;
    RAISE NOTICE 'Updated % rows in config.cnft_form for Ticket.Accordions.Description wrapping', updated_count;
END $do$;

COMMIT;
